#ifndef __CThread__
#define __CThread__

#include "IThread.hpp"
#include "CCriticalSectionScope.hpp"
using Exponent::Threading::CCriticalSectionScope;
using Exponent::Threading::IThread;

//	===========================================================================

namespace Exponent
{
	namespace Threading
	{
		/**
		 * @class CThread CThread.hpp
		 * @brief Implementation of a wrapper around a cross platform thread
		 *
		 * @date 13/01/2006
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CThread.hpp,v 1.3 2007/02/08 21:06:44 paul Exp $
		 */
		class CThread : public IThread, public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

			/**
			 * Construction
			 */
			CThread();

			/**
			 * Destruction
			 */
			virtual ~CThread();

//	===========================================================================

			/**
			 * Do the actual process
			 * @note In subclasses, override this method, but *always* call this\n
			 * base class member first. At the end of your process call, make sure\n
			 * you call CThread::endThread(). If this is your direct base class, the\n
			 * return type is always true and can be ignored
			 * @retval bool True if you completed properly, false otherwise
			 */
			virtual bool runThread();

			/**
			 * Stop the thread
			 * @note This is a forced termination, upon reciept of this message you must terminate as quickly as possible
			 */
			virtual void stopThread();

			/**
			 * Is thread active
			 * @retval bool True if thread is running, false otherwise
			 */
			virtual bool isThreadActive() const;

//	===========================================================================
			  
			/**
			 * Register the thread listener
			 * @param listener The thread listener
			 */
			virtual void registerThreadListener(IThreadListener *listener);

			/**
			 * Get the thread listener
			 * @retval IThreadListener* The thread listener
			 */
			virtual IThreadListener *getThreadListener();

//	===========================================================================

			/**
			 * Set the thread priority
			 * @param priority The thread priority
			 */
			virtual void setThreadPriority(const IThread::EThreadPriority priority);

			/**
			 * Get the thread priority
			 * @return EThreadPriority The thread priority
			 */
			virtual IThread::EThreadPriority getThreadPriority() const;

//	===========================================================================

			/**
			 * Sleep the thread
			 * @param timeInMilliseconds The sleep time in msec
			 */
			void sleepThread(const long timeInMilliseconds);

//	===========================================================================

			/**
			 * Set the Thread handle
			 * @param threadHandle The handle to the thread
			 */
			virtual void setThreadHandle(IThread::SThreadHandle *threadHandle);

			/**
			 * Get the thread handle
			 * @retval IThread::SThreadHandle * Handle to the thread
			 */
			virtual IThread::SThreadHandle *getThreadHandle();

//	===========================================================================

			/**
			 * Set the critical section
			 * @param criticalSection The ciritical setion
			 */
			virtual void setCriticalSection(CCriticalSection *criticalSection);

//	===========================================================================

		protected:

//	===========================================================================

			/**
			 * End the thread
			 * @note Makes sure the thread activity status is set correctly, you must call this function\n
			 * at the end of your process call
			 */
			void endThread();

//	===========================================================================

			EThreadPriority m_priority;							/**< Priority of the this thread */
			IThreadListener *m_threadListener;					/**< Listener for thread events */
			bool m_threadIsActive;								/**< Is the tread active */
			IThread::SThreadHandle *m_threadHandle;				/**< handle to the platform specific thread handle */
			CCriticalSection *m_threadCriticalSection;			/**< Threads own critical section */
			CCriticalSectionScope m_criticalSectionScope;		/**< The critical section scope */
		};
	}
}

#endif		// End of CThread.hpp